<?php
/**
 * Create Employee API
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../config/auth.php';

// Require admin login
requireAdminLogin();

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!isset($input['name']) || !isset($input['department'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Name and department are required']);
    exit();
}

$name = sanitize($input['name']);
$department = sanitize($input['department']);

// Validate input
if (empty($name) || empty($department)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Name and department cannot be empty']);
    exit();
}

if (strlen($name) > 255 || strlen($department) > 255) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Name and department must be less than 255 characters']);
    exit();
}

try {
    $pdo = getDB();
    
    // Generate unique token
    $token = generateToken();
    
    // Ensure token is unique (very unlikely collision, but safety first)
    $stmt = $pdo->prepare("SELECT id FROM employees WHERE qr_token = ?");
    do {
        $stmt->execute([$token]);
        if ($stmt->fetch()) {
            $token = generateToken();
        } else {
            break;
        }
    } while (true);
    
    // Insert employee
    $stmt = $pdo->prepare("INSERT INTO employees (name, department, qr_token) VALUES (?, ?, ?)");
    $stmt->execute([$name, $department, $token]);
    
    $employeeId = $pdo->lastInsertId();
    
    // Get the created employee
    $stmt = $pdo->prepare("SELECT id, name, department, qr_token, created_at FROM employees WHERE id = ?");
    $stmt->execute([$employeeId]);
    $employee = $stmt->fetch();
    
    echo json_encode([
        'success' => true,
        'message' => 'Employee created successfully',
        'employee' => $employee
    ]);
} catch (PDOException $e) {
    error_log("Create employee error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Failed to create employee. Please try again.']);
}
